<?php

/**
 * Sends Messages using the MailerQ server.
 *
 * @author     http://mailerq.com
 */
class MailerQTransport implements Swift_Transport
{
    /** Swift_MailerQConnection that handles publishing the message */
    private $_connection;
    
    /** Additional options to send along with the message */
    private $_options;
    
    /**
     * Constructs the MailerQTransport class.
     * 
     * @param Swift_MailerQConnection $connection
     */ 
    public function __construct(MailerQConnection $connection, $options = array())
    {
        // core parameters
        $this->_connection = $connection;
        $this->_options = $options;
    }
    
    /**
     * Determines whether the connection
     * has been established or not
     */
    public function isStarted()
    {
        return $this->_connection->valid();
    }

    /**
     * Not used.
     */
    public function start() { }

    /**
     * Not used.
     */
    public function stop() { }

    /**
     * Send the given Message.
     *
     * Recipient/sender data will be retrieved from the Message API.
     * The return value is the number of recipients who were accepted for delivery.
     *
     * @param Swift_Mime_Message $message
     * @param string[]           $failedRecipients An array of failures by-reference
     *
     * @return integer
     */
    public function send(Swift_Mime_Message $message, &$failedRecipients = null)
    {
        // merge all recipients into one array
        $recipients = array_keys(array_merge(
            $this->_getTo($message), 
            $this->_getCC($message), 
            $this->_getBCC($message)
        ));
        
        // if any of recipients has not been found, throw an error
        if (!$recipients) throw new Swift_TransportException('Cannot send a message without a recipient');        
        
        // and publish the message to all recipients
        foreach ($recipients as $recipient)
        {
            $this->_connection->publishMessage(array_merge(array(
                'envelope'  => $this->_getEnvelope($message),
                'recipient' => $recipient,
                'body'      =>  $message->toString()
            ), $this->_options));
        }
    }
    
    public function registerPlugin(Swift_Events_EventListener $plugin)
    {
       // not used
    }
    
    /**
     * 
     * Returns the envelope address from the message.
     * 
     * @param Swift_Mime_Message $message
     */
    private function _getEnvelope(Swift_Mime_Message $message)
    {
        $return = $message->getReturnPath();
        $reply = $message->getReplyTo();
        $from = $message->getFrom();
        $envelope = null;
        
        if (!empty($return)) {
            $envelope = $return;
        } elseif (!empty($reply)) {
            $keys = array_keys($reply);
            $envelope = array_shift($keys);
        } elseif (!empty($from)) {
            $keys = array_keys($from);
            $envelope = array_shift($keys);
        }

        return $envelope;
    }
    
    /**
     * Helper method to retrieve TO from the message
     * 
     * @param Swift_Mime_Message $message
     */
    private function _getTo(Swift_Mime_Message $message)
    {
        $to = $message->getTo();
        return (!empty($to)) ? $to : array();
    }
    
    /**
     * Helper method to retrieve CC from the message
     * 
     * @param Swift_Mime_Message $message
     */
    private function _getCC(Swift_Mime_Message $message)
    {
        $cc = $message->getCC();
        return (!empty($cc)) ? $cc : array();
    }    
    
    /**
     * Helper method to retrieve BCC from the message
     * 
     * @param Swift_Mime_Message $message
     */    
    private function _getBCC(Swift_Mime_Message $message)
    {
        $bcc = $message->getBCC();
        return (!empty($bcc)) ? $bcc : array();        
    }
}
